<?php
// auth/login.php
include '../config.php';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Verificar token CSRF
    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        http_response_code(403);
        die('Token CSRF inválido');
    }

    $email = sanitize_input($_POST['email']);
    $password = $_POST['password'];
    
    // Validaciones
    if (empty($email) || empty($password)) {
        $error = "Todos los campos son requeridos";
    } elseif (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $error = "Email inválido";
    } else {
        $database = new Database();
        $db = $database->getConnection();
        
        try {
            // Buscar usuario con prevención de timing attacks
            $query = "SELECT * FROM usuarios WHERE email = :email AND estado = 'activo' LIMIT 1";
            $stmt = $db->prepare($query);
            $stmt->bindParam(':email', $email);
            $stmt->execute();
            
            if ($stmt->rowCount() == 1) {
                $user = $stmt->fetch(PDO::FETCH_ASSOC);
                
                // Verificar contraseña de forma segura
                if (password_verify($password, $user['password'])) {
                    // Regenerar ID de sesión después del login
                    session_regenerate_id(true);
                    
                    // Establecer datos de sesión
                    $_SESSION['user_id'] = $user['id'];
                    $_SESSION['user_name'] = $user['nombre'];
                    $_SESSION['user_email'] = $user['email'];
                    $_SESSION['user_type'] = $user['tipo'];
                    $_SESSION['user_phone'] = $user['telefono'];
                    $_SESSION['user_balance'] = $user['saldo'];
                    $_SESSION['user_rating'] = $user['rating'];
                    $_SESSION['user_trips'] = $user['total_viajes'];
                    $_SESSION['user_contacts'] = $user['total_contactos'];
                    $_SESSION['logged_in'] = true;
                    $_SESSION['login_time'] = time();
                    
                    // Actualizar última conexión
                    $update_query = "UPDATE usuarios SET ultima_conexion = NOW() WHERE id = :id";
                    $update_stmt = $db->prepare($update_query);
                    $update_stmt->bindParam(':id', $user['id']);
                    $update_stmt->execute();
                    
                    // Redirigir según tipo de usuario
                    if ($user['tipo'] == 'admin') {
                        header("Location: ../admin/index.php");
                    } elseif ($user['tipo'] == 'conductor') {
                        header("Location: ../conductor/index.php");
                    } else {
                        header("Location: ../dashboard/index.php");
                    }
                    exit();
                    
                } else {
                    // Usar mismo tiempo para evitar timing attacks
                    password_verify(random_bytes(32), $user['password']);
                    $error = "Credenciales incorrectas";
                }
            } else {
                // Usar mismo tiempo para evitar timing attacks
                password_verify(random_bytes(32), '$2y$10$' . random_bytes(53));
                $error = "Credenciales incorrectas";
            }
        } catch (PDOException $e) {
            error_log("Login error: " . $e->getMessage());
            $error = "Error del sistema. Intente más tarde.";
        }
    }
}

// Si hay error o método GET, mostrar formulario
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>TreyNov - Iniciar Sesión</title>
    <link rel="stylesheet" href="../assets/css/__1.css">
    <link rel="stylesheet" href="../assets/css/small.css">
    <style>
        .login-container {
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            background: linear-gradient(135deg, #0b1822, #1a2d3e);
            padding: 20px;
        }

        .login-card {
            background: rgba(255, 255, 255, 0.1);
            backdrop-filter: blur(10px);
            border-radius: 15px;
            padding: 30px;
            width: 100%;
            max-width: 400px;
            border: 1px solid rgba(255, 255, 255, 0.2);
            box-shadow: 0 8px 32px rgba(0,0,0,0.3);
        }

        .login-logo {
            text-align: center;
            margin-bottom: 30px;
        }

        .login-logo img {
            width: 80px;
            height: 80px;
            border-radius: 50%;
        }

        .login-logo h1 {
            color: white;
            margin-top: 10px;
            font-size: 24px;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            display: block;
            color: white;
            margin-bottom: 5px;
            font-weight: bold;
        }

        .form-group input {
            width: 100%;
            padding: 12px;
            border: none;
            border-radius: 8px;
            background: rgba(255, 255, 255, 0.9);
            font-size: 16px;
            box-sizing: border-box;
            transition: background 0.3s;
        }

        .form-group input:focus {
            background: white;
            outline: none;
            box-shadow: 0 0 0 2px #37B3FF;
        }

        .login-btn {
            width: 100%;
            padding: 12px;
            background: #37B3FF;
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: bold;
            cursor: pointer;
            transition: background 0.3s;
        }

        .login-btn:hover {
            background: #2a8fcc;
        }

        .login-btn:active {
            transform: translateY(1px);
        }

        .error {
            background: #ff4444;
            color: white;
            padding: 12px;
            border-radius: 8px;
            margin-bottom: 15px;
            text-align: center;
            border-left: 4px solid #cc0000;
        }

        .register-link {
            text-align: center;
            margin-top: 20px;
            color: white;
        }

        .register-link a {
            color: #37B3FF;
            text-decoration: none;
            font-weight: bold;
        }

        .register-link a:hover {
            text-decoration: underline;
        }

        .demo-accounts {
            background: rgba(255,255,255,0.1);
            padding: 15px;
            border-radius: 8px;
            margin-top: 20px;
            color: white;
            font-size: 14px;
        }

        .demo-accounts h4 {
            margin: 0 0 10px 0;
            color: #37B3FF;
        }
    </style>
</head>
<body>
    <div class="login-container">
        <div class="login-card">
            <div class="login-logo">
                <img src="../assets/img/icon.png" alt="TreyNov Logo">
                <h1>TreyNov</h1>
                <p style="color: #ccc; margin-top: 5px;">Viaja Seguro</p>
            </div>

            <?php if (isset($error)): ?>
                <div class="error"><?php echo $error; ?></div>
            <?php endif; ?>

            <form method="POST" action="" id="loginForm">
                <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                
                <div class="form-group">
                    <label for="email">Email:</label>
                    <input type="email" id="email" name="email" required 
                           value="<?php echo isset($_POST['email']) ? htmlspecialchars($_POST['email']) : ''; ?>">
                </div>

                <div class="form-group">
                    <label for="password">Contraseña:</label>
                    <input type="password" id="password" name="password" required 
                           minlength="6" autocomplete="current-password">
                </div>

                <button type="submit" class="login-btn" id="submitBtn">Iniciar Sesión</button>
            </form>

            <div class="register-link">
                ¿No tienes cuenta? <a href="register.php">Regístrate aquí</a>
            </div>

            <div class="demo-accounts">
                <h4>👑 Cuentas de Prueba:</h4>
                <strong>Usuario:</strong> frank@example.com / password<br>
                <strong>Conductor:</strong> jorge@example.com / password<br>
                <strong>Admin:</strong> admin@treynov.com / password
            </div>
        </div>
    </div>

    <script>
        // Prevenir múltiples envíos
        document.getElementById('loginForm').addEventListener('submit', function(e) {
            const submitBtn = document.getElementById('submitBtn');
            submitBtn.disabled = true;
            submitBtn.textContent = 'Iniciando sesión...';
            submitBtn.style.background = '#666';
        });

        // Validación en tiempo real
        document.getElementById('email').addEventListener('blur', function() {
            if (!this.value.match(/^[^\s@]+@[^\s@]+\.[^\s@]+$/)) {
                this.style.border = '2px solid #ff4444';
            } else {
                this.style.border = 'none';
            }
        });
    </script>
</body>
</html>