<?php
// api/notifications.php
include '../config.php';
include '../auth_check.php';

header('Content-Type: application/json');

class NotificationSystem {
    private $db;
    
    public function __construct($database) {
        $this->db = $database->getConnection();
    }
    
    // Crear notificación
    public function createNotification($user_id, $title, $message, $type = 'info', $data = null) {
        try {
            $query = "INSERT INTO notificaciones 
                     (usuario_id, titulo, mensaje, tipo, datos, leida, fecha_creacion) 
                     VALUES 
                     (:usuario_id, :titulo, :mensaje, :tipo, :datos, 0, NOW())";
            
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(':usuario_id', $user_id);
            $stmt->bindParam(':titulo', $title);
            $stmt->bindParam(':mensaje', $message);
            $stmt->bindParam(':tipo', $type);
            $stmt->bindParam(':datos', $data);
            
            return $stmt->execute();
        } catch (PDOException $e) {
            error_log("Create notification error: " . $e->getMessage());
            return false;
        }
    }
    
    // Obtener notificaciones del usuario
    public function getUserNotifications($user_id, $limit = 20, $offset = 0) {
        try {
            $query = "SELECT * FROM notificaciones 
                     WHERE usuario_id = :usuario_id 
                     ORDER BY fecha_creacion DESC 
                     LIMIT :limit OFFSET :offset";
            
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(':usuario_id', $user_id);
            $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
            $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
            $stmt->execute();
            
            return $stmt->fetchAll(PDO::FETCH_ASSOC);
        } catch (PDOException $e) {
            error_log("Get notifications error: " . $e->getMessage());
            return [];
        }
    }
    
    // Marcar notificación como leída
    public function markAsRead($notification_id, $user_id) {
        try {
            $query = "UPDATE notificaciones SET leida = 1, fecha_leida = NOW() 
                     WHERE id = :id AND usuario_id = :usuario_id";
            
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(':id', $notification_id);
            $stmt->bindParam(':usuario_id', $user_id);
            
            return $stmt->execute();
        } catch (PDOException $e) {
            error_log("Mark notification as read error: " . $e->getMessage());
            return false;
        }
    }
    
    // Obtener conteo de notificaciones no leídas
    public function getUnreadCount($user_id) {
        try {
            $query = "SELECT COUNT(*) as count FROM notificaciones 
                     WHERE usuario_id = :usuario_id AND leida = 0";
            
            $stmt = $this->db->prepare($query);
            $stmt->bindParam(':usuario_id', $user_id);
            $stmt->execute();
            
            return $stmt->fetch(PDO::FETCH_ASSOC)['count'];
        } catch (PDOException $e) {
            error_log("Get unread count error: " . $e->getMessage());
            return 0;
        }
    }
}

// Crear tabla de notificaciones si no existe
$create_table_query = "
CREATE TABLE IF NOT EXISTS notificaciones (
    id INT PRIMARY KEY AUTO_INCREMENT,
    usuario_id INT NOT NULL,
    titulo VARCHAR(255) NOT NULL,
    mensaje TEXT NOT NULL,
    tipo ENUM('info', 'success', 'warning', 'error') DEFAULT 'info',
    datos JSON NULL,
    leida BOOLEAN DEFAULT FALSE,
    fecha_creacion TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    fecha_leida TIMESTAMP NULL,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE CASCADE,
    INDEX idx_usuario_fecha (usuario_id, fecha_creacion),
    INDEX idx_leida (leida)
)";

$database = new Database();
$db = $database->getConnection();
$db->exec($create_table_query);

$notificationSystem = new NotificationSystem($database);

// Manejar diferentes métodos
$method = $_SERVER['REQUEST_METHOD'];

switch ($method) {
    case 'GET':
        $action = $_GET['action'] ?? 'list';
        
        switch ($action) {
            case 'list':
                $limit = intval($_GET['limit'] ?? 20);
                $offset = intval($_GET['offset'] ?? 0);
                $notifications = $notificationSystem->getUserNotifications($_SESSION['user_id'], $limit, $offset);
                echo json_encode(['success' => true, 'notifications' => $notifications]);
                break;
                
            case 'unread_count':
                $count = $notificationSystem->getUnreadCount($_SESSION['user_id']);
                echo json_encode(['success' => true, 'count' => $count]);
                break;
                
            default:
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Acción no válida']);
        }
        break;
        
    case 'POST':
        $data = json_decode(file_get_contents('php://input'), true);
        $action = $data['action'] ?? '';
        
        switch ($action) {
            case 'mark_read':
                if (isset($data['notification_id'])) {
                    $success = $notificationSystem->markAsRead($data['notification_id'], $_SESSION['user_id']);
                    echo json_encode(['success' => $success]);
                } else {
                    http_response_code(400);
                    echo json_encode(['success' => false, 'error' => 'ID de notificación requerido']);
                }
                break;
                
            case 'mark_all_read':
                // Implementar marcar todas como leídas
                echo json_encode(['success' => true, 'message' => 'Todas marcadas como leídas']);
                break;
                
            default:
                http_response_code(400);
                echo json_encode(['success' => false, 'error' => 'Acción no válida']);
        }
        break;
        
    default:
        http_response_code(405);
        echo json_encode(['success' => false, 'error' => 'Método no permitido']);
}
?>