<?php
// auth/register.php
include '../config.php';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Verificar token CSRF
    if (!verify_csrf_token($_POST['csrf_token'] ?? '')) {
        http_response_code(403);
        die('Token CSRF inválido');
    }

    $nombre = sanitize_input($_POST['nombre']);
    $email = sanitize_input($_POST['email']);
    $password = $_POST['password'];
    $confirm_password = $_POST['confirm_password'];
    $telefono = sanitize_input($_POST['telefono']);
    
    // Validaciones
    $errors = [];
    
    if (empty($nombre) || strlen($nombre) < 2) {
        $errors[] = "El nombre debe tener al menos 2 caracteres";
    }
    
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        $errors[] = "Email inválido";
    }
    
    if (strlen($password) < 6) {
        $errors[] = "La contraseña debe tener al menos 6 caracteres";
    }
    
    if ($password !== $confirm_password) {
        $errors[] = "Las contraseñas no coinciden";
    }
    
    if (empty($telefono) || strlen($telefono) < 9) {
        $errors[] = "Teléfono inválido";
    }
    
    if (empty($errors)) {
        $database = new Database();
        $db = $database->getConnection();
        
        try {
            // Verificar si el email ya existe
            $check_query = "SELECT id FROM usuarios WHERE email = :email LIMIT 1";
            $check_stmt = $db->prepare($check_query);
            $check_stmt->bindParam(':email', $email);
            $check_stmt->execute();
            
            if ($check_stmt->rowCount() > 0) {
                $errors[] = "El email ya está registrado";
            } else {
                // Insertar nuevo usuario
                $insert_query = "INSERT INTO usuarios (nombre, email, password, telefono) 
                               VALUES (:nombre, :email, :password, :telefono)";
                $insert_stmt = $db->prepare($insert_query);
                
                $hashed_password = password_hash($password, PASSWORD_DEFAULT);
                
                $insert_stmt->bindParam(':nombre', $nombre);
                $insert_stmt->bindParam(':email', $email);
                $insert_stmt->bindParam(':password', $hashed_password);
                $insert_stmt->bindParam(':telefono', $telefono);
                
                if ($insert_stmt->execute()) {
                    // Enviar email de bienvenida (simulado)
                    // mail($email, "Bienvenido a TreyNov", "Tu cuenta ha sido creada exitosamente");
                    
                    $_SESSION['success'] = "Cuenta creada exitosamente. Ya puedes iniciar sesión.";
                    header("Location: login.php?registered=1");
                    exit();
                } else {
                    $errors[] = "Error al crear la cuenta. Intente nuevamente.";
                }
            }
        } catch (PDOException $e) {
            error_log("Register error: " . $e->getMessage());
            $errors[] = "Error del sistema. Intente más tarde.";
        }
    }
    
    if (!empty($errors)) {
        $error = implode("<br>", $errors);
    }
}
?>
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>TreyNov - Registrarse</title>
    <link rel="stylesheet" href="../assets/css/__1.css">
    <link rel="stylesheet" href="../assets/css/small.css">
    <style>
        .login-container {
            display: flex;
            flex-direction: column;
            justify-content: center;
            align-items: center;
            min-height: 100vh;
            background: linear-gradient(135deg, #0b1822, #1a2d3e);
            padding: 20px;
        }

        .login-card {
            background: rgba(255, 255, 255, 0.1);
            backdrop-filter: blur(10px);
            border-radius: 15px;
            padding: 30px;
            width: 100%;
            max-width: 400px;
            border: 1px solid rgba(255, 255, 255, 0.2);
            box-shadow: 0 8px 32px rgba(0,0,0,0.3);
        }

        .login-logo {
            text-align: center;
            margin-bottom: 30px;
        }

        .login-logo img {
            width: 80px;
            height: 80px;
            border-radius: 50%;
        }

        .login-logo h1 {
            color: white;
            margin-top: 10px;
            font-size: 24px;
        }

        .form-group {
            margin-bottom: 20px;
        }

        .form-group label {
            display: block;
            color: white;
            margin-bottom: 5px;
            font-weight: bold;
        }

        .form-group input {
            width: 100%;
            padding: 12px;
            border: none;
            border-radius: 8px;
            background: rgba(255, 255, 255, 0.9);
            font-size: 16px;
            box-sizing: border-box;
            transition: background 0.3s;
        }

        .form-group input:focus {
            background: white;
            outline: none;
            box-shadow: 0 0 0 2px #37B3FF;
        }

        .password-strength {
            height: 4px;
            background: #ddd;
            border-radius: 2px;
            margin-top: 5px;
            overflow: hidden;
        }

        .strength-0 { width: 0%; background: #ff4444; }
        .strength-1 { width: 25%; background: #ff4444; }
        .strength-2 { width: 50%; background: #ff9800; }
        .strength-3 { width: 75%; background: #4CAF50; }
        .strength-4 { width: 100%; background: #4CAF50; }

        .login-btn {
            width: 100%;
            padding: 12px;
            background: #4CAF50;
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 16px;
            font-weight: bold;
            cursor: pointer;
            transition: background 0.3s;
        }

        .login-btn:hover {
            background: #45a049;
        }

        .login-btn:active {
            transform: translateY(1px);
        }

        .error {
            background: #ff4444;
            color: white;
            padding: 12px;
            border-radius: 8px;
            margin-bottom: 15px;
            text-align: center;
            border-left: 4px solid #cc0000;
        }

        .login-link {
            text-align: center;
            margin-top: 20px;
            color: white;
        }

        .login-link a {
            color: #37B3FF;
            text-decoration: none;
            font-weight: bold;
        }

        .login-link a:hover {
            text-decoration: underline;
        }
    </style>
</head>
<body>
    <div class="login-container">
        <div class="login-card">
            <div class="login-logo">
                <img src="../assets/img/icon.png" alt="TreyNov Logo">
                <h1>Crear Cuenta</h1>
                <p style="color: #ccc; margin-top: 5px;">Únete a TreyNov</p>
            </div>

            <?php if (isset($error)): ?>
                <div class="error"><?php echo $error; ?></div>
            <?php endif; ?>

            <form method="POST" action="" id="registerForm">
                <input type="hidden" name="csrf_token" value="<?php echo generate_csrf_token(); ?>">
                
                <div class="form-group">
                    <label for="nombre">Nombre completo:</label>
                    <input type="text" id="nombre" name="nombre" required 
                           value="<?php echo isset($_POST['nombre']) ? htmlspecialchars($_POST['nombre']) : ''; ?>"
                           minlength="2" maxlength="100">
                </div>

                <div class="form-group">
                    <label for="email">Email:</label>
                    <input type="email" id="email" name="email" required 
                           value="<?php echo isset($_POST['email']) ? htmlspecialchars($_POST['email']) : ''; ?>">
                </div>

                <div class="form-group">
                    <label for="telefono">Teléfono:</label>
                    <input type="tel" id="telefono" name="telefono" required 
                           value="<?php echo isset($_POST['telefono']) ? htmlspecialchars($_POST['telefono']) : ''; ?>"
                           pattern="[0-9]{9,15}" title="Mínimo 9 dígitos">
                </div>

                <div class="form-group">
                    <label for="password">Contraseña:</label>
                    <input type="password" id="password" name="password" required 
                           minlength="6" autocomplete="new-password">
                    <div class="password-strength" id="passwordStrength"></div>
                </div>

                <div class="form-group">
                    <label for="confirm_password">Confirmar Contraseña:</label>
                    <input type="password" id="confirm_password" name="confirm_password" required 
                           minlength="6" autocomplete="new-password">
                    <small id="passwordMatch" style="color: #ccc;"></small>
                </div>

                <button type="submit" class="login-btn" id="submitBtn">Crear Cuenta</button>
            </form>

            <div class="login-link">
                ¿Ya tienes cuenta? <a href="login.php">Inicia sesión aquí</a>
            </div>
        </div>
    </div>

    <script>
        // Validación de fortaleza de contraseña
        document.getElementById('password').addEventListener('input', function() {
            const password = this.value;
            const strengthBar = document.getElementById('passwordStrength');
            let strength = 0;
            
            if (password.length >= 6) strength++;
            if (password.match(/[a-z]/) && password.match(/[A-Z]/)) strength++;
            if (password.match(/\d/)) strength++;
            if (password.match(/[^a-zA-Z\d]/)) strength++;
            
            strengthBar.className = 'password-strength strength-' + strength;
        });

        // Validación de coincidencia de contraseñas
        document.getElementById('confirm_password').addEventListener('input', function() {
            const password = document.getElementById('password').value;
            const confirm = this.value;
            const matchText = document.getElementById('passwordMatch');
            
            if (confirm.length > 0) {
                if (password === confirm) {
                    matchText.textContent = '✓ Las contraseñas coinciden';
                    matchText.style.color = '#4CAF50';
                } else {
                    matchText.textContent = '✗ Las contraseñas no coinciden';
                    matchText.style.color = '#ff4444';
                }
            } else {
                matchText.textContent = '';
            }
        });

        // Prevenir múltiples envíos
        document.getElementById('registerForm').addEventListener('submit', function(e) {
            const submitBtn = document.getElementById('submitBtn');
            submitBtn.disabled = true;
            submitBtn.textContent = 'Creando cuenta...';
            submitBtn.style.background = '#666';
        });

        // Validación de email
        document.getElementById('email').addEventListener('blur', function() {
            if (!this.value.match(/^[^\s@]+@[^\s@]+\.[^\s@]+$/)) {
                this.style.border = '2px solid #ff4444';
            } else {
                this.style.border = 'none';
            }
        });
    </script>
</body>
</html>