<?php
// api/get_conductores.php
include '../config.php';
include '../auth_check.php';

header('Content-Type: application/json');
header('Access-Control-Allow-Origin: ' . $_SERVER['HTTP_ORIGIN'] ?? '*');
header('Access-Control-Allow-Credentials: true');

// Validar método
if ($_SERVER['REQUEST_METHOD'] !== 'GET') {
    http_response_code(405);
    echo json_encode(['success' => false, 'error' => 'Método no permitido']);
    exit();
}

$database = new Database();
$db = $database->getConnection();

// Obtener parámetros con valores por defecto
$tipo = isset($_GET['tipo']) ? sanitize_input($_GET['tipo']) : 'privado';
$ubicacion = isset($_GET['ubicacion']) ? sanitize_input($_GET['ubicacion']) : 'Cajabamba';
$limit = isset($_GET['limit']) ? intval($_GET['limit']) : 20;
$offset = isset($_GET['offset']) ? intval($_GET['offset']) : 0;

// Validar parámetros
if (!in_array($tipo, ['privado', 'grupo'])) {
    echo json_encode(['success' => false, 'error' => 'Tipo de conductor inválido']);
    exit();
}

if ($limit > 50) $limit = 50;
if ($offset < 0) $offset = 0;

try {
    // Consulta optimizada para conductores verificados con plan activo
    $query = "SELECT 
                c.id,
                c.tipo,
                c.vehiculo_modelo,
                c.placa,
                c.licencia,
                c.rating,
                c.total_viajes,
                c.ubicacion_actual,
                c.precio_min,
                c.precio_max,
                c.estado as conductor_estado,
                c.metodo_pago,
                u.nombre,
                u.telefono,
                u.rating as user_rating,
                ce.viajes_exitosos,
                ce.viajes_fallidos,
                ce.reportes,
                ce.kilometros_recorridos,
                ce.horas_conduccion,
                ce.transacciones,
                ce.ganancias_totales,
                pm.plan_tipo,
                pm.fecha_fin as plan_fecha_fin,
                DATEDIFF(pm.fecha_fin, CURDATE()) as dias_restantes_plan
              FROM conductores c
              JOIN usuarios u ON c.usuario_id = u.id
              LEFT JOIN conductor_estadisticas ce ON c.id = ce.conductor_id
              INNER JOIN planes_mensuales pm ON c.id = pm.conductor_id 
              WHERE c.tipo = :tipo 
                AND c.estado = 'disponible'
                AND c.verificado = 'aprobado'
                AND c.ubicacion_actual LIKE :ubicacion
                AND pm.estado = 'activo'
                AND pm.fecha_fin >= CURDATE()
                AND u.estado = 'activo'
              ORDER BY 
                c.rating DESC,
                c.total_viajes DESC,
                pm.fecha_fin DESC
              LIMIT :limit OFFSET :offset";
    
    $stmt = $db->prepare($query);
    $stmt->bindValue(':tipo', $tipo);
    $stmt->bindValue(':ubicacion', '%' . $ubicacion . '%');
    $stmt->bindValue(':limit', $limit, PDO::PARAM_INT);
    $stmt->bindValue(':offset', $offset, PDO::PARAM_INT);
    $stmt->execute();
    
    $conductores = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Obtener total para paginación
    $count_query = "SELECT COUNT(*) as total
                   FROM conductores c
                   JOIN usuarios u ON c.usuario_id = u.id
                   INNER JOIN planes_mensuales pm ON c.id = pm.conductor_id 
                   WHERE c.tipo = :tipo 
                     AND c.estado = 'disponible'
                     AND c.verificado = 'aprobado'
                     AND c.ubicacion_actual LIKE :ubicacion
                     AND pm.estado = 'activo'
                     AND pm.fecha_fin >= CURDATE()";
    
    $count_stmt = $db->prepare($count_query);
    $count_stmt->bindValue(':tipo', $tipo);
    $count_stmt->bindValue(':ubicacion', '%' . $ubicacion . '%');
    $count_stmt->execute();
    $total = $count_stmt->fetch(PDO::FETCH_ASSOC)['total'];
    
    // Formatear respuesta
    foreach ($conductores as &$conductor) {
        $conductor['metodo_pago'] = explode(',', $conductor['metodo_pago']);
        $conductor['precio_min'] = floatval($conductor['precio_min']);
        $conductor['precio_max'] = floatval($conductor['precio_max']);
        $conductor['rating'] = floatval($conductor['rating']);
        $conductor['ganancias_totales'] = floatval($conductor['ganancias_totales']);
    }
    
    echo json_encode([
        'success' => true,
        'conductores' => $conductores,
        'pagination' => [
            'total' => intval($total),
            'limit' => $limit,
            'offset' => $offset,
            'has_more' => ($offset + count($conductores)) < $total
        ],
        'filters' => [
            'tipo' => $tipo,
            'ubicacion' => $ubicacion,
            'requisitos' => [
                'verificado' => true,
                'plan_activo' => true,
                'disponible' => true
            ]
        ]
    ]);
    
} catch (PDOException $e) {
    error_log("API Conductores error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'error' => 'Error interno del servidor'
    ]);
}
?>